////////////////////////////////////////////////////////////////////////////////
//
//  rtUtils.h - some common utils for real time tasks
//  Copyright (C) 2000 Philip N Daly <pnd@noao.edu>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
//  Authors:		    Philip N Daly
//  Contact:		    pnd@noao.edu
//  Version:            v0.1.0
//  Modification date:	Tue, 22 February 2000
//  Description:	    This code module contains some common utilities
//                      for real time Linux tasks. Much of this is taken
//                      out of K&R's C book.
//                      Please send additions/edits to me.
//  Prototypes:
//                      unsigned long rtrand(void);
//                      unsigned long rtatoi(char *);
//                      void *rtstrip(char *);
//                      void rtstrncpy(char *, char *, int n);
//                      void rtstrcpy(char *, char *);
//                      int rtstrlen(char *);
//                      void rtstrcat(char *, char *);
//                      unsigned char rtstrcmp(char *, char *);
//                      char *rtstrchr(char *, char);
//                      volatile void *rtmemset(volatile void *, char, long);
//                      void rt_pr2scr(char *);
//                      LOG-macros
//
////////////////////////////////////////////////////////////////////////////////
#ifndef RTUTILS_H
#define RTUTILS_H
#endif

////////////////////////////////////////////////////////////////////////////////
// some defines (which might or might not be used below)
////////////////////////////////////////////////////////////////////////////////
#define TRUE         0x01
#define YES          0x01
#define FALSE        0x00
#define NO           0x00
#define SWAP(a,b)    {float tempr; tempr=(a); (a)=(b); (b)=tempr;}
#define SQR(a)       ((a)*(a))
#define ABS(x)       (((x)<0)?-(x):(x))
#define MAX(x,y)     (((x)<(y))?(y):(x))
#define MIN(x,y)     (((x)<(y))?(x):(y))
#define FOREVER      for (;;)

////////////////////////////////////////////////////////////////////////////////
// these assume the ASCII collating sequence
////////////////////////////////////////////////////////////////////////////////
#define IS_NULL(c)   ((c)==0x0)
#define IS_WHITE(c)  ((c)<=0x20)
#define IS_QUOTE(c)  ((c)==0x22||(c)==0x27)
#define IS_DIGIT(c)  ((c)>='0'&&(c)<='9')
#define IS_LOWER(c)  ((c)>='a'&&(c)<='z')
#define IS_UPPER(c)  ((c)>='A'&&(c)<='Z')
#define IS_LETTER(c) (((c)>='a'&&(c)<='z')||((c)>='A'&&(c)<='Z'))
#define TO_LOWER(c)  (((c)>='A'&&(c)<='Z')?(c)+0x20:(c))
#define TO_UPPER(c)  (((c)>='a'&&(c)<='z')?(c)-0x20:(c))

////////////////////////////////////////////////////////////////////////////////
// a pseudo random number generator
////////////////////////////////////////////////////////////////////////////////
#ifdef NEED_RTSEED
static unsigned long rtseed = 0;
unsigned long rtrand( void ) {
  rtseed = rtseed * 0x41C64E6D + 0x3039;
  return (rtseed>>16)&0x7FFF;
}
#endif

////////////////////////////////////////////////////////////////////////////////
// ascii to integer (cf. atoi)
////////////////////////////////////////////////////////////////////////////////
#ifdef NEED_RTATOI
unsigned long rtatoi( char *s ) {
  unsigned long iRtn  = (unsigned long) 0;
  int first = TRUE;
  while ( IS_DIGIT(*s) ) {
    if ( first ) {
      first = FALSE;
      iRtn = 0;
    }
    iRtn = iRtn*10 + *(s++) - '0';
  }
  return (iRtn);
}
#endif

////////////////////////////////////////////////////////////////////////////////
// ascii to integer (cf. strtol)
////////////////////////////////////////////////////////////////////////////////
#ifdef NEED_RTSTRTOL
unsigned long rtstrtol( char *s, char **e, int base ) {
  unsigned long iRtn  = (unsigned long) 0;
  char c;
  while((c = TO_LOWER(*s)) && (IS_DIGIT(c) || IS_LETTER(c))){
    s++;
    iRtn = iRtn * base + (c - (IS_DIGIT(c) ? '0' : 'a'-10));
  }
  if(e) *e=s;
  return (iRtn);
}
#endif

////////////////////////////////////////////////////////////////////////////////
// cleans whitespace and quote-marks from input string
////////////////////////////////////////////////////////////////////////////////
#ifdef NEED_RTSTRIP
void *rtstrip( char *s ) {
  char *p = s;
  int   i = 0;
  while ( ! IS_NULL(*s) ) {
    if ( IS_QUOTE(*s) ) *s = 0x20;
    if ( ! IS_WHITE(*s) ) {
      i++;
      *p = TO_LOWER(*s);
      p++;
    }
    s++;
  }
  *p = '\0';
  return (p-i);
}
#endif

////////////////////////////////////////////////////////////////////////////////
// copies n character from t to s (cf. strncpy)
////////////////////////////////////////////////////////////////////////////////
#ifdef NEED_RTSTRNCPY
void rtstrncpy( char *s, char *t, int n ) {
  int i = 0;
  while ( (i<ABS(n)) && (*s++=*t++) ) {i++;}
}
#endif

////////////////////////////////////////////////////////////////////////////////
// copies t to s (cf. strcpy)
////////////////////////////////////////////////////////////////////////////////
#ifdef NEED_RTSTRCPY
void rtstrcpy( char *s, char *t ) {
  while ( (*s++=*t++) ) {;}
}
#endif

////////////////////////////////////////////////////////////////////////////////
// returns the length of the input string (cf. strlen)
////////////////////////////////////////////////////////////////////////////////
#ifdef NEED_RTSTRLEN
int rtstrlen( char *s ) {
  char *p = s;
  while ( *p != '\0' ) {p++;}
  return (p - s);
}
#endif

////////////////////////////////////////////////////////////////////////////////
// concatenates t to s (cf. strcat)
////////////////////////////////////////////////////////////////////////////////
#ifdef NEED_RTSTRCAT
void rtstrcat( char *s, char *t ) {
  int slen = rtstrlen(s);
  char *p  = s + slen;
  while ( (*p++=*t++) ) {;}
}
#endif

////////////////////////////////////////////////////////////////////////////////
// compares two string (cf. strcmp)
////////////////////////////////////////////////////////////////////////////////
#ifdef NEED_RTSTRCMP
unsigned char rtstrcmp( char *s, char *t ) {
  unsigned char satisfied = TRUE;
  int slen                = rtstrlen(s);
  int tlen                = rtstrlen(t);
  if ( slen!=tlen || slen<=0 || tlen<=0 ) satisfied = FALSE;
  while ( satisfied ) {
    if ( IS_NULL(s) && IS_NULL(*t) ) break;
    if ( *s == *t ) {
      s++;
      t++;
    } else {
      satisfied = FALSE;
    }
  }
  return (satisfied);
}
#endif

////////////////////////////////////////////////////////////////////////////////
// finds the given character within the given string (cf. strchr)
////////////////////////////////////////////////////////////////////////////////
#ifdef NEED_RTSTRCHR
char *rtstrchr( char *s, char c ) {
  while ( *s != c ) {
    if ( *s == '\0' ) return (NULL);
    s++;
  }
  return (s);
}
#endif

////////////////////////////////////////////////////////////////////////////////
// fills the array with the given character (cf. memset)
////////////////////////////////////////////////////////////////////////////////
#ifdef NEED_RTMEMSET
volatile void *rtmemset( volatile void *s, char c, long size ) {
 long i = 0;
 volatile char *p = s;
 while ( i<size ) {
   *p=c; i++; p++;
 }
 return (s);
}
#endif

////////////////////////////////////////////////////////////////////////////////
// prints pre-formatted string to current tty
////////////////////////////////////////////////////////////////////////////////
#ifdef NEED_PR2SCR
#include <linux/sched.h>
#include <linux/tty.h>
int rt_pr2scr( char *s ) {
#ifdef __KERNEL__
 int slen = rtstrlen(s);
#else
 int slen = strlen(s);
#endif
 struct tty_struct *ltty = current->tty;
 if ( ltty != NULL ) {
  (*(ltty->driver).write) (ltty,0,s,slen);        // write the string out
  (*(ltty->driver).write) (ltty,0,"\015\012",2);  // add <CR><LF>
 }
 // return number of characters written
 return (slen+2);
}
#endif

////////////////////////////////////////////////////////////////////////////////
// LOG-macros
////////////////////////////////////////////////////////////////////////////////
#ifdef NEED_LOGMAC
#define MSGLEN 80
typedef int (*PFV)(char *);
#ifdef __KERNEL__
 #ifdef DEBUG
  // NB: output will be double spaced
  static PFV msgFunc = (PFV) rt_pr2scr;
 #endif
#else // can use these macros from the user side too
 #ifdef DEBUG
  // NB: output will be double spaced
  static PFV msgFunc = (PFV) rt_pr2scr;
 #endif
#endif
// define as many of these as you like 
#define LOG(x) { char msg[MSGLEN];\
 (void) sprintf(msg,"%s\n",(x));\
 (void) (msgFunc) (msg);}
#define LOGC(x,y) { char msg[MSGLEN];\
 (void) sprintf(msg,"%s %c\n",(x),(char)(y));\
 (void) (msgFunc) (msg);}
#define LOGS(x,y) { char msg[MSGLEN];\
 (void) sprintf(msg,"%s %s\n",(x),(y));\
 (void) (msgFunc) (msg);}
#define LOGF(x,y) { char msg[MSGLEN];\
 (void) sprintf(msg,"%s %f\n",(x),(float)(y));\
 (void) (msgFunc) (msg);}
#define LOGI(x,y) { char msg[MSGLEN];\
 (void) sprintf(msg,"%s %d\n",(x),(int)(y));\
 (void) (msgFunc) (msg);}
#define LOGL(x,y) { char msg[MSGLEN];\
 (void) sprintf(msg,"%s %ld\n",(x),(long)(y));\
 (void) (msgFunc) (msg);}
#define LOGX(x,y) { char msg[MSGLEN];\
 (void) sprintf(msg,"%s %#2x\n",(x),(int)(y));\
 (void) (msgFunc) (msg);}
#endif
