/*********************************************************
 *                                                       *
 *                  Can4Opener                           *
 *                                                       *
 *********************************************************
 *                                                       *
 *********************************************************
 * This program is free software; you can redistribute   *
 * it and/or modify it under the terms of the GNU General*
 * Public License as published by the Free Software      *
 * Foundation; either version 2 of the License, or (at   *
 * your option) any later version.                       *
 *                                                       *
 * This program is distributed in the hope that it will  *
 * be useful, but WITHOUT ANY WARRANTY; without even the *
 * implied warranty of MERCHANTABILITY or FITNESS FOR A  *
 * PARTICULAR PURPOSE.  See the GNU General Public       *
 * License for more details.                             *
 *                                                       *
 * You should have received a copy of the GNU General    *
 * Public License along with this program; if not, write *
 * to 	The Free Software Foundation, Inc.               *
 *	675 Mass Ave                                     *
 *	Cambridge                                        *
 *	MA 02139                                         *
 * 	USA.                                             *
 *********************************************************
 *                                                       *
 *      Author: Nathan Gustavson, based on the CanOpener *
 *      application by Edouard Tisserant                 *
 *                                                       *
 *      Contact: edouard.tisserant@esstin.u-nancy.fr     *
 *      Version: 1.1                                     *
 *      Modification date:12/9/02                        *
 *      Mod Author N.Z. Gustavson ngustavson@emacinc.com *
 *      Description:                                     *
 *      Added locks to prevent conflicts due to new      * 
 *	Asychronous Notification scheme                  *
 *-------------------------------------------------------*
 * A powerfull textmode tool to explore a CanOpen bus.   *
 * with modifications to use the can4linux hardware layer*
 *                                                       *
 *********************************************************/ 

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/fcntl.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <curses.h>

#include "Can.h"
#include "CanOpenMatic.h"
#include "CanOpener.h"
#include "Can4Open.h"


#define MAX_ID 128

// Link to CanOpenMatic proceed_infos
extern proceed_info proceed_infos[];

int busfd;
char busname[40];
extern int wait;
volatile int datawait=0;
static struct {
	int N;
	int l;
	int n[MAX_ID];
	} stats[16] = {[0 ... 15] = {0, 0,{[0 ... MAX_ID-1] = 0}}};
WINDOW *win;
WINDOW  *swin1, *swin2, *swin3;

void drawstats(){
	int fc;
	int i;
	int stat_line = 1;
	static char str[20] = "Stats";

	werase(swin2);
	box(swin2, ACS_VLINE, ACS_HLINE);
	mvwaddstr(swin2, 0, 3, str);
	for(fc = 0; fc<16; fc++){
		int k=0;
		for ( i = 0; i < MAX_ID ;i++){
			if(stats[fc].N > 0)
				mvwprintw(swin2, stat_line, 1, "%-6.6s(%d)",proceed_infos[fc].name, stats[fc].N);
			if(stats[fc].n[i]>0){
				k++;
				mvwprintw(swin2, stat_line + k/4, 8 + (k%4)*7, "%d:%d",i, stats[fc].n[i]);
			}
			stats[fc].l = (k+3)/4;
		}

		if(k>MAX_ID/8){ //Scan ping filter
			k=0;
			sprintf(str,"Stat (filtered)");
			for ( i = 0; i < MAX_ID ;i++)
				if(stats[fc].n[i]<=1)
					stats[fc].n[i]=0;
				else
					k++;
			stats[fc].l = (k+4)/4;
		}
	stat_line+=stats[fc].l;
	}
	wrefresh(swin2);
}

volatile int loglock;

void can_logrx(Message *m, int state){

//while(loglock);
 // sleep(1);//wait for logging function to unlock
				  
can_log(m,state);

datawait=0;
}


void can_logtx(Message *m, int state){



}




void can_log(Message *m, int state){

	int x,y;
	int i;
	loglock = 1;
	getmaxyx(swin1, y, x);
	scroll(swin1);
	if(state>=0){
		int fc = GET_FUNCTION_CODE((*m));
		int ID = GET_NODE_ID((*m));
		mvwprintw(swin1,
			y-2, 1,
			"%s%-6.6s %2.2x %s",
			state == STATE_Tx ? ">" : "<",
			proceed_infos[fc].name,
			ID,
			m->rtr ? "Yes" : "No ");
		for ( i = 0; i < m->len;i++){
			wprintw(swin1, "%2.2x ",m->data[i]);
		}
		stats[fc].n[ID]++;
		stats[fc].N++;
	} else {
		mvwprintw(swin1, y-2, 1, "Read Error (%d)",state);
	}
	mvwvline(swin1,y-2,0,ACS_VLINE,1);
	mvwvline(swin1,y-2,x-1,ACS_VLINE,1);

	wrefresh(swin1);

	drawstats();
loglock = 0;
}




void help()
{
	printf("**************************************************************\n");
	printf("*  Can4Opener -b b                                            *\n");
	printf("*                [-t t -s -c c -n n -w -i i -f f -l l]       *\n");
	printf("*		                                             *\n");
	printf("*	t : period					     *\n");
	printf("*	b : bus						     *\n");
	printf("*	s : send sync each cycle			     *\n");
	printf("*	n : number responses expected after all		     *\n");
	printf("*	w : wait for all responses			     *\n");
	printf("*	i : number of iterations [infinity]		     *\n");
	printf("*	f : first counter value	[0]			     *\n");
	printf("*	l : last counter value	[i]			     *\n");
	printf("*	c : emit content string				     *\n");
	printf("*		Sd[iii|cs|d0|d1|d2|d3|d4|d5|d6]w : SDO       *\n");
	printf("*		Pd[iii|d0|d1|d2|d3|d4|d5|d6|d7]w : PDO       *\n");
	printf("*		 d: direction ('r' for Rx or 't' for Tx)     *\n");
	printf("*		 iii: ID	(xxx for counter)	     *\n");
	printf("*		 cs : SCS Sdo Command Specifier		     *\n");
	printf("*		Nd[iii|st]w  (direction ignoerd) : NMT       *\n");
	printf("*		 d: direction ('r' for read or 'w' for write)*\n");
	printf("*		 iii: ID	(xxx for counter)	     *\n");
	printf("*		 st : NMT state				     *\n");
	printf("*		Ed[iii|i0|i1|si|d0|d1|d2|..|dn]w : dic Entry *\n");
	printf("*		 i0 - i1 : index (short, i0=LSB)             *\n");
	printf("*		 si : sub-index                              *\n");
	printf("*		                                             *\n");
	printf("*		 d0 - dn : data, if 'xn'=>counter byte (opt) *\n");
	printf("*		 w : listen w message BEFORE send (opt)      *\n");
	printf("*		                                             *\n");
	printf("*  This exemple sniff bus 0 (wait each messages)             *\n");
	printf("*   Can4Opener -b can0 -w -n1                               *\n");
	printf("*		                                             *\n");
	printf("*  This exemple send sync and PDORx with counter and const   *\n");
	printf("*	for ID 1 on bus 0 whith 10ms period                  *\n");
	printf("*   Can4Opener -b can0 -t 10000 -s -c \"Pr[001|x0|x1|03]\"  *\n");
	printf("**************************************************************\n");
}                                                           


int WinStart()
{
int     w, h, sw, sh, bx, by;

	initscr();
	start_color();
	cbreak();
	if ((win = newwin(LINES, COLS, 0, 0))==NULL) {
		endwin();
		return 1;
	}
	init_pair(1, COLOR_WHITE, COLOR_BLACK);
	wattrset(win, COLOR_PAIR(1));
	wbkgd(win, COLOR_PAIR(1));
	werase(win);
	box(win, ACS_VLINE, ACS_HLINE);
	mvwaddstr(win, 0, 3, "Can4Opener v1.0");
	mvwaddstr(win, 0, COLS-19, "hit ctrl-c to exit");
	wrefresh(win);

	nodelay(win,TRUE);

	getmaxyx(win, h,  w);
	getbegyx(win, by, bx);
	sw = w / 2;
	sh = 4;
	if((swin1 = subwin(win, h-2, sw, by+1, bx+sw-1)) == NULL)
		return  1;
	if((swin2 = subwin(win, h-sh-2, sw-2, 1, bx+1)) == NULL)
		return  1;
	if((swin3 = subwin(win, sh, sw-2, h-sh-1, bx+1)) == NULL)
		return  1;

	init_pair(2,COLOR_WHITE,COLOR_BLUE);
	wattrset(swin1, COLOR_PAIR(2));
	wbkgd(swin1, COLOR_PAIR(2));
	werase(swin1);
	box(swin1, ACS_VLINE, ACS_HLINE);
	mvwaddstr(swin1, 0, 3, "Sniff");
	mvwaddstr(swin1, 1, 1, "Type   ID RTR        Data");
	wsetscrreg(swin1, 2, h - 4);
	scrollok(swin1, TRUE);
	wmove(swin1, 2, 3);
	wrefresh(swin1);

	init_pair(3,COLOR_WHITE,COLOR_BLUE);
	wattrset(swin2, COLOR_PAIR(3));
	wbkgd(swin2, COLOR_PAIR(3));
	werase(swin2);
	box(swin2, ACS_VLINE, ACS_HLINE);
	mvwaddstr(swin2, 0, 3, "Stats");
	wrefresh(swin2);

	init_pair(4,COLOR_WHITE,COLOR_BLUE);
	wattrset(swin3, COLOR_PAIR(4));
	wbkgd(swin3, COLOR_PAIR(4));
	werase(swin3);
	box(swin3, ACS_VLINE, ACS_HLINE);
	mvwaddstr(swin3, 0, 3, "Cycle");
	wrefresh(swin3);


	wrefresh(win);

	return  0;
}

int WinStop()
{
	delwin(swin1);
	delwin(swin2);
	delwin(swin3);
	delwin(win);
	endwin();
	return  0;
}

int SendFromCmd(char *cmd, int ssync, int cf, int cl, int iter, int responses, int period)
{
  int i,a=0;
  extern logfunc rxloggingfunc,txloggingfunc;
  rxloggingfunc = can_logrx;
  //txloggingfunc = can_log; /*don't log tx messages*/

  loglock = 0;
			       
  do{
    char *beg = cmd + 1;
    char *end = beg;
    int c = (cf+(((cl-cf)*a)/iter));
    Message m;
    
    mvwprintw(swin3, 1, 1, "Iteration %d/%d    ", a,iter);
    mvwprintw(swin3, 2, 1, "Counter   %d<%d<%d    ", cf, c, cl);
    wrefresh(swin3);
    
    if(ssync)
      Send_Sync(0);
    
    while((beg = index(end, '[')) && (end = index(beg, ']'))){
      unsigned char data[100];
      int ID = 0;
      int i = 0;
      char *msgt = beg - 2;
      int msgd = *(beg - 1) == 'r' ? Rx : Tx;
      beg+=1;
      
      if(*beg=='x')
	ID = c ;
      else
	ID = strtol(beg,NULL,16);
      
      for(beg+=4;beg<end && i<9; beg+=3){
	if(*beg=='x' && i<5){
	  int dec = (*(beg+1)-'0') % 4;
	  data[i++]=(c & (0xff << (8*dec)))>>(8*dec) ;
	}
	else
	  data[i++] = strtol(beg,NULL,16);
      };
      
      //if(*(end + 1) == 'w')
	//f_can_receive(0, &m);
      
      //if(wait)
        //pause();

      switch(*msgt)
	{
	case 'S' :
	  {
	    s_SDO sdo;
	    sdo.ID = ID;
	    sdo.len = i<9 ? i : 8;
	    memcpy(&(sdo.body), &(data[0]), sdo.len);
	    
	    Send_SDO(0, sdo, msgd);					   					   					
	  }
	break;
	case 'P' :
	  {
	    s_PDO pdo;
	    pdo.ID = ID;
	    pdo.len =  i<9 ? i : 8;
	    memcpy(&(pdo.data), &(data[0]), pdo.len);
	    
	    Send_PDO(0, pdo, msgd);
	  }
	break;
	case 'N' :
	  {
	    
	    Send_NMT(0, ID, data[0]);
	  }
	break;
	case 'E' :
	  {
	    if(i>=3){
	      int tmpwait = wait;
	      //wait = 1;	// This kind of protocol need waiting
			     if(msgd==Tx)
							WriteDictionaryEntry(0, ID, *((short *)&data[0]), data[2], i-3, &data[3]);
			     else
			       ReadDictionaryEntry(0, ID, *((short *)&data[0]), data[2], &data[3]);
	      wait = tmpwait;
	    }
	  }
	break;
	default:
	  return -1;
	}

    }
    
    //for(i=0; i<responses; i++)
      //f_can_receive(0, &m);

    if(wait)
      pause();

	
    if(period >0) usleep(period);
    
  }while(++a != iter && wgetch(win)==ERR);
	return                                                                                                                      0;
}

int main(int argc,char **argv)
{
	int ssync = 0;
	int period = -1;
	int responses = -1;
	int iter = -1;
	int cf = 0;
	int cl = -1;

	char c;
	extern char *optarg;
	char cmd[CMD_SIZE] = {[0 ... CMD_SIZE-1] = 0};

	if (argc == 1)
		{
		help();
		exit(1);
		}

	while ((c = getopt(argc, argv, "-t:b:sn:c:wi:f:l:")) != EOF)
		{
		switch(c)
			{
			case 't' :
				period = atoi(optarg);
				break;
			case 'b' :
				sprintf(busname,"/dev/%s", optarg);
				break;
			case 's' :
				ssync = 1;
				break;
			case 'n' :
				responses = atoi(optarg);
				break;
			case 'w' :
				wait = 1;
				break;
			case 'c' :
				sprintf(cmd,"%s", optarg);
				break;
			case 'i' :
				iter = atoi(optarg);
				break;
			case 'f' :
				cf = atoi(optarg);
				break;
			case 'l' :
				cl = atoi(optarg);
				break;
			default:
				help();
				exit(1);
			}
		}

	if(cl < 0)
		cl=iter;
	else if(iter<0)
		iter=cl-cf;


	if(busname[0] == 0)
		{
		help();
		exit(1);
		}

	if((busfd = f_can_open(busname))<0){
		perror("Unable to open device");
		exit(1);
	}

	if(WinStart()){
		perror("Probleme with curses");
		WinStop();
		exit(1);
	}

	if(SendFromCmd(cmd,ssync, cf, cl, iter, responses, period)){;
		WinStop();
		help();
		exit(1);
	} else
		WinStop();
	f_can_close(busfd);
	return 0;
}


